<?php
session_start();

include 'db.php';

if (!isset($_SESSION['username'])) {
    header("Location: login.php");
    exit;
}

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); // biar error kelihatan
$conn = new mysqli($servername, $username, $password, $database);
$conn->set_charset("utf8mb4");

$email = trim($_SESSION['username'] ?? '');
if ($email === '') {
    die("Sesi tidak valid: email kosong.");
}

/**
 * Helper: ambil satu baris/NULL
 */
function fetch_one(mysqli $conn, string $sql, string $types = '', array $params = []) : ?array {
    $stmt = $conn->prepare($sql);
    if ($types !== '' && $params) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $res  = $stmt->get_result();
    $row  = $res->fetch_assoc() ?: null;
    $stmt->close();
    return $row;
}

/** 1) Hitung total & ambil no_plg dengan sekali query (lebih efisien) */
$sqlTot = "
    SELECT 
        t.no_plg,
        SUM( t.Jumlah_beli * (CASE WHEN b.Diskon>0 
                                   THEN (b.Harga - b.Harga*(b.Diskon/100)) 
                                   ELSE b.Harga END)
        ) AS total_bayar
    FROM transaksi t
    JOIN barang b    ON t.no_brg = b.no_id
    JOIN pelanggan p ON t.no_plg = p.no_id
    WHERE p.Email = ? AND t.Invoice IS NULL
";
$agg = fetch_one($conn, $sqlTot, "s", [$email]);

if (!$agg || !$agg['no_plg']) {
    echo "Tidak ada transaksi yang bisa di-checkout.";
    exit;
}

$no_plg       = (int)$agg['no_plg'];
$tot_jenderal = (float)$agg['total_bayar'];

if ($tot_jenderal <= 0) {
    echo "Anda belum membeli apa-apa.";
    exit;
}

$_SESSION['tot_jenderal'] = $tot_jenderal;
$_SESSION['no_plg']       = $no_plg;

/** 2) Buat nomor invoice unik + update baris transaksi (atomic) */
$conn->begin_transaction();
try {
    do {
        try {
            $randpart = random_int(0, 9999);
        } catch (Exception $e) {
            $randpart = mt_rand(0, 9999);
        }
        $invoice = "INV" . date('Ymd') . sprintf("%04d", $randpart);

        // coba update; kalau gagal karena uniq constraint, ulangi
        $upd = $conn->prepare("UPDATE transaksi SET Invoice = ? WHERE no_plg = ? AND Invoice IS NULL");
        $upd->bind_param("si", $invoice, $no_plg);
        $upd->execute();
        $affected = $upd->affected_rows;
        $upd->close();

        if ($affected > 0) {
            // verifikasi tidak bentrok (kalau ada UNIQUE KEY, ini aman)
            break;
        } else {
            // tidak ada baris diupdate -> kemungkinan user klik dua kali
            throw new RuntimeException("Gagal membuat invoice. Mungkin transaksi sudah di-checkout sebelumnya.");
        }
    } while (true);

    $conn->commit();
} catch (Throwable $e) {
    $conn->rollback();
    die("Gagal membuat invoice: " . htmlspecialchars($e->getMessage()));
}

/** 3) Ambil data rekening / email bukti / ongkos kirim dengan aman */
$rek = fetch_one($conn, "SELECT Nama_bank, No_rek, Atas_nama FROM rekening LIMIT 1");
$buk = fetch_one($conn, "SELECT Email_bukti FROM bukti LIMIT 1");
$kir = fetch_one($conn, "SELECT Ongkos FROM kirim LIMIT 1");

$nama_bank = $rek['Nama_bank'] ?? '';
$no_rek    = $rek['No_rek'] ?? '';
$atas_nama = $rek['Atas_nama'] ?? '';
$email_resmi = $buk['Email_bukti'] ?? '';
$ongkos      = isset($kir['Ongkos']) ? (float)$kir['Ongkos'] : 0.0;

// fallback teks agar tidak tampil “kosong”
if ($nama_bank === '' || $no_rek === '' || $atas_nama === '') {
    $nama_bank = $nama_bank !== '' ? $nama_bank : "(belum diset admin)";
    $no_rek    = $no_rek    !== '' ? $no_rek    : "(belum diset admin)";
    $atas_nama = $atas_nama !== '' ? $atas_nama : "(belum diset admin)";
}
if ($email_resmi === '') {
    $email_resmi = "(belum diset admin)";
}

$conn->close();
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Check Out</title>
    <style>
        body { font-family: Verdana, Arial, sans-serif; }
        .wrap { max-width: 640px; margin: 40px auto; line-height: 1.6; }
        .box { border: 1px solid #ddd; padding: 16px; border-radius: 8px; }
        .total { font-weight: bold; }
        .muted { color: #666; }
        .danger { color: #b00020; }
        a.btn { display:inline-block; padding:8px 14px; border:1px solid #222; border-radius:6px; text-decoration:none; }
    </style>
</head>
<body>
<div class="wrap">
    <div class="box">
        <p>Anda sudah Check Out.</p>
        <p>Nomor INVOICE Anda : <b><?php echo htmlspecialchars($invoice); ?></b></p>

        <p>Jumlah Total SEMUA yang harus dibayar:<br>
           Rp. <?php echo number_format($tot_jenderal, 0, ',', '.'); ?></p>

        <p>Dengan Ongkos kirim Rp. <?php echo number_format($ongkos, 0, ',', '.'); ?></p>

        <p class="total">Jadi jumlah yang harus dibayar:
           <b>Rp. <?php echo number_format($tot_jenderal + $ongkos, 0, ',', '.'); ?></b></p>

        <hr>
        <p>Silakan kirim pembayaran ke:</p>
        <p>
            Nama Bank: <?php echo htmlspecialchars($nama_bank); ?><br>
            No. Rek: <?php echo htmlspecialchars($no_rek); ?><br>
            a/n: <?php echo htmlspecialchars($atas_nama); ?>
        </p>

        <?php
        // beri peringatan jika data belum diset
        if (strpos($nama_bank, 'belum diset') !== false || strpos($no_rek, 'belum diset') !== false || strpos($atas_nama, 'belum diset') !== false) {
            echo '<p class="danger"><b>PERINGATAN:</b> Data rekening belum diset oleh admin. Silakan hubungi admin.</p>';
        }
        ?>

        <p>Kirim bukti pembayaran ke email kami: <?php echo htmlspecialchars($email_resmi); ?></p>
        <?php
        if (strpos($email_resmi, 'belum diset') !== false) {
            echo '<p class="danger"><b>PERINGATAN:</b> Email tujuan bukti pembayaran belum diset admin.</p>';
        }
        ?>

        <p><a class="btn" href="dashboard.php">Selesai</a></p>
        <p class="muted">Simpan nomor invoice Anda untuk referensi.</p>
    </div>
</div>
</body>
</html>
